<?php

namespace Drupal\instnt_pattern_library;

use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Provides the default Pattern Manager.
 */
class PatternManager {

  /**
   * The Theme Handler instance.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * The File System instance to operate on files.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Array with Pattern instances.
   *
   * @var array
   */
  protected $patterns = [];

  /**
   * Creates an instance of a Pattern.
   *
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler
   *   Theme handler.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   File system.
   */
  public function __construct(ThemeHandlerInterface $theme_handler, FileSystemInterface $file_system) {
    $this->themeHandler = $theme_handler;
    $this->fileSystem = $file_system;
  }

  /**
   * Method to load the patterns.
   *
   * @return array
   *   An empty array if no patterns were found or
   *   an array of Pattern instances.
   */
  public function loadPatterns() {
    // Check if patterns already loaded.
    if (empty($this->patterns)) {
      $theme_name = $this->themeHandler->getDefault();
      $directory = $this->themeHandler->getTheme($theme_name)->getPath();

      if (!is_dir($directory)) {
        return [];
      }

      $depth = count(explode('/', $directory));

      // Get a list of all yaml files inside the patterns directory. We
      // use 'min_depth' to ensure that a pattern exists within a collection.
      $files = array_keys($this->fileSystem->scanDirectory($directory, '/.*\.pattern\.yml$/', ['min_depth' => 2]));

      // Sort files alphabetically.
      sort($files);

      foreach ($files as $file) {
        $name = basename($file, '.pattern.yml');

        // Retrieve the base path.
        $base_path = array_slice(explode('/', $file), $depth, -1);
        $base_path = implode('/', $base_path);

        // Check if the pattern is located in the default templates folder.
        if (substr($base_path, 0, 10) === 'templates/') {
          $base_path = '@' . $theme_name . '/' . substr($base_path, 10);
        }

        // Create a Pattern instance.
        $pattern = new Pattern($name, $file, $base_path);

        // Push the Pattern instance to the patterns array.
        $this->patterns[] = $pattern;
      }
    }

    return $this->patterns;
  }

  /**
   * Retrieve an array with all the Pattern instances.
   *
   * @return array
   *   Returns an array of Pattern instances.
   */
  public function getPatterns() {
    return $this->loadPatterns();
  }

  /**
   * Retrieve a single Pattern instance.
   *
   * @param string $collection
   *   Name of pattern.
   * @param string $name
   *   Name of pattern.
   *
   * @return null|\Drupal\instnt_pattern_library\Pattern
   *   A Pattern instance or NULL.
   */
  public function getPattern(string $collection, string $name) {
    $patterns = $this->loadPatterns();

    $pattern = array_filter($patterns, function ($pattern) use ($collection, $name) {
      return $pattern->getCollection() === $collection && $pattern->getName() === $name;
    });

    $pattern = array_values($pattern);

    return reset($pattern) ?? NULL;
  }

  /**
   * Retrieve the first Pattern instance.
   *
   * @return \Drupal\instnt_pattern_library\Pattern
   *   A Pattern instance.
   */
  public function getFirst() {
    $patterns = $this->loadPatterns();

    return reset($patterns) ?: NULL;
  }

}
