<?php

namespace Drupal\instnt_pattern_library;

use Drupal\Core\Extension\InfoParserException;
use Symfony\Component\Yaml\Yaml;

/**
 * Defines a single Pattern instance.
 *
 * @package Drupal\instnt_pattern_library
 */
class Pattern {

  /**
   * Pattern prefix.
   */
  const PATTERN_PREFIX = 'pattern_';

  /**
   * Pattern name.
   *
   * @var string
   */
  protected $name;

  /**
   * Pattern yaml file.
   *
   * @var string
   */
  protected $file;

  /**
   * Pattern base path.
   *
   * @var string
   */
  protected $basePath;

  /**
   * Pattern label.
   *
   * @var string
   */
  protected $label;

  /**
   * Pattern description.
   *
   * @var string
   */
  protected $description;

  /**
   * Bool to show meta information.
   *
   * @var bool
   */
  protected $showMeta;

  /**
   * Template name.
   *
   * @var string|null
   */
  protected $template;

  /**
   * Pattern scheme.
   *
   * @var array
   */
  protected $scheme;

  /**
   * Pattern data.
   *
   * @var array
   */
  protected $data;

  /**
   * Pattern implementation code.
   *
   * @var string
   */
  protected $code;

  /**
   * Component constructor.
   *
   * @param string $name
   *   Identifier of the component.
   * @param string $file
   *   Path of the yaml file.
   * @param string $base_path
   *   The base path of the Pattern.
   *
   * @throws \UnexpectedValueException
   *   Exception thrown if a collection cannot be determined from a file.
   */
  public function __construct(string $name, string $file, string $base_path) {
    $this->name = $name;
    $this->file = $file;
    $this->basePath = $base_path;

    // Determine collection from filename.
    $parts = explode('/', dirname($file));
    $this->collection = array_slice($parts, -2, 1)[0] ?? '';

    if (empty($this->collection)) {
      throw new \UnexpectedValueException("Can't determine collection from file.");
    }

    $this->parseYamlFile($file);
  }

  /**
   * Method to parse the yaml file of a pattern and set the contents.
   *
   * @param string $file
   *   Path of file.
   *
   * @throws \Drupal\Core\Extension\InfoParserException
   *   Exception thrown if there is a parsing error or the .yml file does
   *   not contain a required key.
   */
  protected function parseYamlFile($file) {
    // Parse the file.
    $contents = Yaml::parseFile($file);

    // Check if label exists and is not empty.
    if (empty($contents['label'])) {
      throw new InfoParserException("Missing required key 'label' or the key is empty.");
    }

    // Set scheme.
    $scheme = [];
    if (isset($contents['scheme']) && is_array($contents['scheme'])) {
      foreach ($contents['scheme'] as $key => $prop) {
        $scheme[] = ['name' => $key] + $prop;
      }
    }

    // Update pattern variables with values parsed from yaml file.
    $this->label = $contents['label'];
    $this->description = $contents['description'] ?? '';
    $this->scheme = $scheme;
    $this->data = isset($contents['data']) && is_array($contents['data']) ? $contents['data'] : [];
    $this->showMeta = filter_var($contents['show_meta'] ?? FALSE, FILTER_VALIDATE_BOOLEAN);
    $this->template = $contents['template'] ?? NULL;
  }

  /**
   * Get the Pattern name.
   *
   * @return mixed
   *   Returns the name.
   */
  public function getName() {
    return $this->name;
  }

  /**
   * Get the label of the Pattern.
   *
   * @return string
   *   Returns the label.
   */
  public function getLabel() {
    return $this->label;
  }

  /**
   * Get the collection name of the Pattern.
   *
   * @return string
   *   The name of the collection.
   */
  public function getCollection() {
    return $this->collection;
  }

  /**
   * Get the path of the Pattern.
   *
   * @return string
   *   The path of the Pattern.
   */
  public function getBasePath() {
    return $this->basePath;
  }

  /**
   * Method to generate the implementation code for the pattern type component.
   *
   * @return string
   *   Returns a html string.
   */
  public function getCode() {
    if (is_null($this->code)) {
      $scheme = $this->getScheme();
      $filename = '\'' . $this->getBasePath() . '/' . $this->getName() . '.html.twig\'';

      // Check if base path is a namespace.
      if (substr($this->getBasePath(), 0, 1) !== '@') {
        $filename = 'directory ~ \'/' . $this->getBasePath() . '/' . $this->getName() . '.html.twig\'';
      }

      // If template is set, use it.
      if ($this->getTemplate()) {
        $filename = '\'' . str_replace('.example', '', $this->getTemplate()) . '\'';
      }

      $html = '{% include ' . $filename;

      if (!empty($scheme)) {
        $variables = '';

        foreach ($scheme as $item) {
          if ($item['required'] ?? FALSE && !empty($item['value'])) {
            $value = json_encode($item['value'], JSON_UNESCAPED_SLASHES);
            $value = str_replace(['\"', '"'], ['&quot;', '\''], addcslashes($value, '\''));
            $variables .= '  ' . '\'' . $item['name'] . '\'' . ': ' . $value . ',<br>';
          }
        }

        if (!empty($variables)) {
          $html .= ' with { <br>' . $variables . '} only';
        }
      }

      $html .= ' %}';

      $this->code = $html;
    }

    return $this->code;
  }

  /**
   * Get Pattern theme implementation.
   *
   * @return array
   *   Returns the pattern theme implementation.
   */
  public function getThemeImplementation() {
    $theme = [
      'template' => $this->getName(),
      'path' => dirname($this->getFile()),
      'variables' => $this->getVariables(),
    ];

    // If specific template is given, use that one.
    if ($this->getTemplate()) {
      $theme['template'] = 'pattern-wrapper';
      $theme['variables'] += ['template_file' => $this->getTemplate()];
      unset($theme['path']);
    }

    return [self::PATTERN_PREFIX . $this->getCollection() . '_' . $this->getName() => $theme];
  }

  /**
   * Get the Pattern file path.
   *
   * @return string
   *   Returns the file path.
   */
  public function getFile() {
    return $this->file;
  }

  /**
   * Get the description of the Pattern.
   *
   * @return string
   *   Returns the description.
   */
  public function getDescription() {
    return $this->description;
  }

  /**
   * Show the meta information of the Pattern or not.
   *
   * @return bool
   *   Returns TRUE when the information must shown.
   */
  public function showMetaInformation() {
    return $this->showMeta;
  }

  /**
   * Get the template name when available.
   *
   * @return string|null
   *   The name of the template or NULL when not exists.
   */
  public function getTemplate() {
    return $this->template;
  }

  /**
   * Get the Pattern scheme.
   *
   * @return array
   *   Returns the scheme.
   */
  public function getScheme() {
    return $this->scheme;
  }

  /**
   * Get the data of the Pattern.
   *
   * @return array
   *   Returns the data.
   */
  public function getData() {
    return $this->data;
  }

  /**
   * Get the variables with NULL values.
   *
   * @return array
   *   Returns the variables.
   */
  public function getVariables() {
    $variables = [];
    $scheme = $this->getScheme();

    foreach ($scheme as $item) {
      if (array_key_exists('name', $item)) {
        $variables[$item['name']] = NULL;
      }
    }

    return $variables;
  }

}
