<?php

namespace Drupal\instnt_pattern_library\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\instnt_pattern_library\Pattern;
use Drupal\instnt_pattern_library\PatternManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Returns a Response with a Pattern.
 *
 * @package Drupal\instnt_pattern_library\Controller
 */
class PatternController extends ControllerBase {

  /**
   * Pattern Manager service.
   *
   * @var Drupal\instnt_pattern_library\PatternManager
   */
  protected $patternManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(PatternManager $patternManager) {
    $this->patternManager = $patternManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('instnt_pattern_library.manager')
    );
  }

  /**
   * Title callback.
   *
   * @param string $collection
   *   The name of the collection.
   * @param string $name
   *   Name of pattern.
   *
   * @return string
   *   The label of the Pattern.
   */
  public function title($collection, $name) {
    return $this->patternManager->getPattern($collection, $name)->getLabel();
  }

  /**
   * Loads the first Pattern and redirect the user.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response object that may be returned by the controller.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   Throwed when pattern is not found.
   */
  public function index() {
    $pattern = $this->patternManager->getFirst();

    if (!$pattern) {
      throw new NotFoundHttpException();
    }

    return $this->redirect('instnt_pattern_library.view', [
      'collection' => $pattern->getCollection(),
      'name' => $pattern->getName(),
    ], [], 301);
  }

  /**
   * Get a single Pattern and return a renderable array.
   *
   * @param string $collection
   *   The name of the collection.
   * @param string $name
   *   Name of pattern.
   *
   * @return array
   *   Returns renderable array.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   Throwed when pattern is not found.
   */
  public function view($collection, $name) {
    $pattern = $this->patternManager->getPattern($collection, $name);

    if (!$pattern) {
      throw new NotFoundHttpException();
    }

    // Get the menu for all the patterns.
    $menu = $this->getPatternMenu();

    return [
      '#theme' => 'pattern_item',
      '#type' => 'page',
      '#pattern' => [
        'name' => $pattern->getName(),
        'label' => $pattern->getLabel(),
        'description' => [
          '#type' => 'processed_text',
          '#text' => $pattern->getDescription(),
          '#format' => filter_default_format(),
        ],
        'show_meta' => $pattern->showMetaInformation(),
        'scheme' => $pattern->getScheme(),
        'variants' => $this->getPatternVariants($pattern),
        'code' => $pattern->getCode(),
      ],
      '#menu' => $menu,
    ];
  }

  /**
   * Get the menu for navigation.
   *
   * @return array
   *   The menu.
   */
  protected function getPatternMenu() {
    $menu = [];
    $patterns = $this->patternManager->getPatterns();

    foreach ($patterns as $pattern) {
      $menu[$pattern->getCollection()][] = [
        'label' => $pattern->getLabel(),
        'url' => Url::fromRoute('instnt_pattern_library.view', [
          'collection' => $pattern->getCollection(),
          'name' => $pattern->getName(),
        ]),
      ];
    }

    return $menu;
  }

  /**
   * Build a render array with variants.
   *
   * @param Drupal\instnt_pattern_library\Pattern $pattern
   *   The Pattern instance.
   *
   * @return array
   *   Render array with variants.
   */
  protected function getPatternVariants(Pattern $pattern) {
    $variants = [];
    $data = $pattern->getData();
    $theme = Pattern::PATTERN_PREFIX . $pattern->getCollection() . '_' . $pattern->getName();

    // Global function to build a render array.
    $build = function ($item) use ($theme) {
      $renderer = [
        '#theme' => $theme,
      ];

      foreach ($item as $key => $value) {
        $renderer['#' . $key] = $value;
      }

      return $renderer;
    };

    // Loop through all the data.
    foreach ($data as $item) {
      // If the item doesn't have variants, build a render array.
      if (!isset($item['variants'])) {
        $variants[] = [
          'variants' => [
            $build($item),
          ],
        ];

        continue;
      }

      // The item has variants. Build a group with render arrays.
      $renderer = [
        'label' => $item['label'] ?? NULL,
        'variants' => [],
      ];

      foreach ($item['variants'] as $variant) {
        $renderer['variants'][] = $build($variant);
      }

      $variants[] = $renderer;
    };

    // If data key is not present, still add one renderer to display template.
    if (empty($variants)) {
      $variants[] = [
        'variants' => [
          $build([]),
        ],
      ];
    }

    return $variants;
  }

}
